﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.IO;
using System.Windows.Media.Imaging;

namespace AvWorksBrowser.Web
{
    /// <summary>
    /// Wyświetla obrazki umieszczone w kolumnie ThumbNailPhoto typu varbinary
    /// tabeli Product bazy AdventureWorks, jako obrazki PNG, przesyłająć je 
    /// przy użyciu protokołu HTTP.
    /// </summary>
    /// <remarks>
    /// Dane obrazka są przechowywane w kolumnie varbinary zapisane w formacie GIF.
    /// To rozwiązanie nie jest przydatne dla klienta Silverlight, gdyż obsługuje
    /// on jedynie formaty PNG i JPEG. (Jedynym powodem dla którego można by się
    /// zastanawiać nad zastosowaniem formatu GIF a nie JPEG, jest chęć skorzystania 
    /// z animacji, ale ponieważ w Silverlight animacje działają na zupełnie innym
    /// poziomie, to takie rozwiązanie i tak by nie działało.
    /// 
    /// Dlatego też konieczne jest przekonwertowanie danych z formatu GIF do PNG,
    /// gdyż klient Silverlight nie będzie w stanie zrobić tego samodzielnie.
    /// Zasoby możemy udostępniać przy użyciu protokołu HTTP, gdyż dzięki temu 
    /// Silverlight będzie je w stanie pobierać i przechowywać w pamięci podręcznej
    /// tak jak wszystkie inne obrazki pobierane z internetu. Oznacza to, 
    /// że w Silverlight 3 możliwe jest asynchroniczne pobieranie i wyświetlanie obrazków.
    /// </remarks>
    public class ProductThumbnail : IHttpHandler
    {
        #region IHttpHandler Members

        public bool IsReusable { get { return true; } }

        public void ProcessRequest(HttpContext context)
        {
            // Expecting a ?id=1234 parameter on the request.
            string idParam = context.Request["id"];
            int id;
            byte[] binData = null;
            string etag = null;
            bool matchesExisting = false;
            DateTime modifiedDate = new DateTime();

            if (idParam != null && int.TryParse(idParam, out id))
            {

                using (var ctx = new AdventureWorksLT2008Entities())
                {
                    var productQ = from product in ctx.Products
                                   where product.ProductID == id
                                   select new
                                   {
                                       product.ThumbNailPhoto,
                                       product.rowguid,
                                       product.ModifiedDate
                                   };

                    var productInfo = productQ.FirstOrDefault();
                    if (productInfo != null)
                    {
                        etag = productInfo.rowguid.ToString("N");
                        modifiedDate = productInfo.ModifiedDate;

                        // Three outcomes: null means If-Modified-Since not present (so ETag is the decider),
                        // true means a match, false means no match.
                        bool? matchesIfModifiedSince = null;

                        // Check If-Modified-Since request header, if it exists 
                        string ifModifiedSince = context.Request.Headers["If-Modified-Since"];
                        DateTime ifModifiedDateTime;
                        if (!string.IsNullOrEmpty(ifModifiedSince) && ifModifiedSince.Length > 0 && DateTime.TryParse(ifModifiedSince, out ifModifiedDateTime))
                        {
                            double modifiedDiff = Math.Abs((ifModifiedDateTime - productInfo.ModifiedDate).TotalSeconds);

                            // Date encoding can give us variability of up to a second, apparently.
                            matchesIfModifiedSince = modifiedDiff < 1.0;
                        }

                        bool? matchesEtag = null;
                        string ifNoneMatch = context.Request.Headers["If-None-Match"];
                        if (!string.IsNullOrEmpty(ifNoneMatch) && ifNoneMatch.Length > 0)
                        {
                            matchesEtag = ifNoneMatch != etag;
                        }

                        // To match, either If-Modified-Since or If-None-Match must be supplied
                        // and must match, and if both are supplied, both must match.
                        // We could use the following monstrosity I wrote when digging into
                        // how the tristate logical binary operators (| and &) work:
                        //if (((matchesIfModifiedSince | matchesEtag) & ((!matchesIfModifiedSince | !matchesEtag) != true)) == true)
                        // But I'd rather read this:
                        bool positiveMatch = (matchesExisting | matchesEtag).GetValueOrDefault();
                        bool definiteMismatch = (!matchesIfModifiedSince | !matchesEtag).GetValueOrDefault();
                        if (positiveMatch && !definiteMismatch)
                        {
                            context.Response.StatusCode = 304;
                            context.Response.StatusDescription = "Not Modified";
                            context.Response.AddHeader("Content-Length", "0");
                            matchesExisting = true;
                        }
                    }

                    binData = productInfo.ThumbNailPhoto;
                }
            }
            if (binData == null)
            {
                context.Response.StatusCode = 404;
                context.Response.StatusDescription = "Not Found";
                context.Response.End();
                return;
            }
            else
            {
                context.Response.Cache.SetCacheability(HttpCacheability.Private);
                context.Response.Cache.VaryByHeaders["If-Modified-Since"] = true;
                context.Response.Cache.VaryByHeaders["If-None-Match"] = true;
                context.Response.Cache.SetLastModified(modifiedDate);
                context.Response.Cache.SetETag(etag);

                if (!matchesExisting)
                {
                    MemoryStream originalBitmap = new MemoryStream(binData);
                    var dec = BitmapDecoder.Create(originalBitmap, BitmapCreateOptions.None, BitmapCacheOption.Default);

                    context.Response.ContentType = "image/png";
                    PngBitmapEncoder pngEncoder = new PngBitmapEncoder();
                    pngEncoder.Frames.Add(dec.Frames[0]);

                    // Can't write directly to the output stream, because this all goes
                    // via WIC. The interop wrappers we get between a .NET Stream and
                    // WIC's idea of a stream end up attempting to to a Stat on the
                    // stream, and ASP.NET throws a NotSupportedException when that
                    // tries to get the stream length. So we write into a temporary
                    // in-memory stream first.

                    MemoryStream pngBitmapStream = new MemoryStream();
                    pngEncoder.Save(pngBitmapStream);

                    byte[] pngBitmapBytes = pngBitmapStream.ToArray();
                    //context.Response.OutputStream.SetLength(pngBitmapBytes.Length);
                    context.Response.OutputStream.Write(pngBitmapBytes, 0, pngBitmapBytes.Length);
                }

            }
        }

        #endregion
    }
}